<?php
namespace ReyCore\Modules\ElementorAcf;

if (!defined('ABSPATH')) exit; // Exit if accessed directly

class Base extends \ReyCore\Modules\ModuleBase {

	public function __construct()
	{

		parent::__construct();

		add_action( 'elementor/widgets/widgets_registered', [ $this, 'register_widgets' ] );
		add_action( 'elementor/elements/categories_registered', [ $this, 'add_elementor_widget_categories'] );
		add_filter( 'reycore/query-control/autocomplete', [$this, 'query_control_autocomplete'], 10, 2);
		add_filter( 'reycore/query-control/values', [$this, 'query_control_values'], 10, 2);
	}

	public function register_widgets( $widgets_manager ) {

		foreach ( [
			'text',
			'heading',
			'button',
			'image',
			// 'icon', // field uses 2 values, icon string and library
		] as $class_name ) {


			$class_name = ucwords( str_replace( '-', ' ', $class_name ) );
			$class_name = str_replace( ' ', '', $class_name );
			$class_name = \ReyCore\Helper::fix_class_name($class_name, 'Modules\\ElementorAcf');

			$widgets_manager->register_widget_type( new $class_name );
		}
	}

	/**
	 * Add Rey Widget Categories
	 *
	 * @since 1.0.0
	 */
	public function add_elementor_widget_categories( $elements_manager ) {

		$categories = [
			'rey-acf' => [
				'title' => __( 'REY - ACF Fields', 'rey-core' ),
				'icon' => 'fa fa-plug',
			],
		];

		foreach( $categories as $key => $data ){
			$elements_manager->add_category($key, $data);
		}
	}

	public static function get_acf_fields( $types = [] ){

		if ( function_exists( 'acf_get_field_groups' ) ) {
			$acf_groups = acf_get_field_groups();
		}

		$default_types = [
			'text',
			'textarea',
			'number',
			'email',
			'wysiwyg',
			'select',
			'checkbox',
			'radio',
			'true_false',
			'oembed',
			'google_map',
			'date_picker',
			'time_picker',
			'date_time_picker',
			'color_picker',
			'image',
		];

		$options = [];

		$options_page_groups_ids = [];

		if ( function_exists( 'acf_options_page' ) ) {

			$pages = acf_options_page()->get_pages();

			foreach ( $pages as $slug => $page ) {

				$options_page_groups = acf_get_field_groups( [
					'options_page' => $slug,
				] );

				foreach ( $options_page_groups as $options_page_group ) {
					$options_page_groups_ids[ $options_page_group['key'] ] = $page['post_id'];
				}

			}
		}

		foreach ( $acf_groups as $acf_group ) {

			if ( function_exists( 'acf_get_fields' ) ) {
				if ( isset( $acf_group['ID'] ) && ! empty( $acf_group['ID'] ) ) {
					$fields = acf_get_fields( $acf_group['ID'] );
				} else {
					$fields = acf_get_fields( $acf_group );
				}
			}


			if ( ! is_array( $fields ) ) {
				continue;
			}

			foreach ( $fields as $field ) {

				if ( ! in_array( $field['type'], $types, true ) ) {
					continue;
				}

				if( array_key_exists($field['parent'], $options_page_groups_ids) ){
					$key = $field['key'] . ':' . $field['name'] . ':' . $options_page_groups_ids[$field['parent']];
				}
				else {
					$key = $field['key'] . ':' . $field['name'];
				}

				$options[ $key ] = sprintf('%s > %s', $acf_group['title'], $field['label']);

			}

			if ( empty( $options ) ) {
				continue;
			}

		}

		return $options;

	}

	function query_control_values($results, $data){

		if( ! isset($data['query_args']['type']) ){
			return $results;
		}

		if( $data['query_args']['type'] !== 'acf' ){
			return $results;
		}

		$field_types = isset($data['query_args']['field_types']) ? $data['query_args']['field_types'] : [];
		$fields = self::get_acf_fields( $field_types );

		foreach ((array) $data['values'] as $id) {
			if( isset($fields[$id]) ){
				$results[ $id ] = $fields[$id];
			}
		}

		return $results;
	}

	function query_control_autocomplete($results, $data){

		if( ! isset($data['query_args']['type']) ){
			return $results;
		}

		if( $data['query_args']['type'] !== 'acf' ){
			return $results;
		}

		$field_types = isset($data['query_args']['field_types']) ? $data['query_args']['field_types'] : [];
		$fields = self::get_acf_fields( $field_types );

		foreach( $fields as $id => $text ){
			if( strpos($id, $data['q']) !== false || strpos(strtolower($text), strtolower($data['q'])) !== false ){
				$results[] = [
					'id' 	=> $id,
					'text' 	=> $text,
				];
			}
		}

		return $results;
	}

	public static function get_field( $key ){

		$parts = explode(':', $key);
		$field = $parts[0];
		$post_id = get_queried_object();

		// has option page
		if( count($parts) > 2 ){
			$post_id = $parts[2];
		}

		if( is_tax() ){
			if( isset($post_id->term_id) && ($tax = $post_id) ){
				$post_id = $tax->term_id;
				// if( reycore__is_multilanguage() ){
				// 	$post_id = apply_filters('reycore/translate_ids', $post_id, $tax->taxonomy );
				// }
			}
			$post_id = get_term_by('term_taxonomy_id', $post_id);
		}
		else {
			// if( reycore__is_multilanguage() ){
			// 	$post_id = apply_filters('reycore/translate_ids', $post_id );
			// }
		}

		return get_field($field, $post_id );
	}

	public static function __config(){
		return [
			'id' => basename(__DIR__),
			'title' => esc_html_x('ACF Fields for Elementor', 'Module name', 'rey-core'),
			'description' => esc_html_x('Adds new Elementor elements (Text, Image, Button, etc) which can render the contents of any Advanced Custom Field is picked. Great tool for dynamic pages.', 'Module description', 'rey-core'),
			'icon'        => '',
			'categories'  => ['elementor'],
			'keywords'    => [''],
			'help'        => reycore__support_url('kb/adding-acf-fields-inside-pages'),
			'video' => true,
		];
	}

	public function module_in_use(){

		$results = \ReyCore\Elementor\Helper::scan_content_in_site( 'element', [
			\ReyCore\Elementor\Widgets::PREFIX . 'acf-button',
			\ReyCore\Elementor\Widgets::PREFIX . 'acf-heading',
			\ReyCore\Elementor\Widgets::PREFIX . 'acf-text',
			\ReyCore\Elementor\Widgets::PREFIX . 'acf-image',
		] );

		return ! empty($results);

	}

}
